﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/networkmanager/NetworkManagerRequest.h>
#include <aws/networkmanager/NetworkManager_EXPORTS.h>
#include <aws/networkmanager/model/CoreNetworkPolicyAlias.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace NetworkManager {
namespace Model {

/**
 */
class GetCoreNetworkPolicyRequest : public NetworkManagerRequest {
 public:
  AWS_NETWORKMANAGER_API GetCoreNetworkPolicyRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetCoreNetworkPolicy"; }

  AWS_NETWORKMANAGER_API Aws::String SerializePayload() const override;

  AWS_NETWORKMANAGER_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The ID of a core network.</p>
   */
  inline const Aws::String& GetCoreNetworkId() const { return m_coreNetworkId; }
  inline bool CoreNetworkIdHasBeenSet() const { return m_coreNetworkIdHasBeenSet; }
  template <typename CoreNetworkIdT = Aws::String>
  void SetCoreNetworkId(CoreNetworkIdT&& value) {
    m_coreNetworkIdHasBeenSet = true;
    m_coreNetworkId = std::forward<CoreNetworkIdT>(value);
  }
  template <typename CoreNetworkIdT = Aws::String>
  GetCoreNetworkPolicyRequest& WithCoreNetworkId(CoreNetworkIdT&& value) {
    SetCoreNetworkId(std::forward<CoreNetworkIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of a core network policy version.</p>
   */
  inline int GetPolicyVersionId() const { return m_policyVersionId; }
  inline bool PolicyVersionIdHasBeenSet() const { return m_policyVersionIdHasBeenSet; }
  inline void SetPolicyVersionId(int value) {
    m_policyVersionIdHasBeenSet = true;
    m_policyVersionId = value;
  }
  inline GetCoreNetworkPolicyRequest& WithPolicyVersionId(int value) {
    SetPolicyVersionId(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The alias of a core network policy </p>
   */
  inline CoreNetworkPolicyAlias GetAlias() const { return m_alias; }
  inline bool AliasHasBeenSet() const { return m_aliasHasBeenSet; }
  inline void SetAlias(CoreNetworkPolicyAlias value) {
    m_aliasHasBeenSet = true;
    m_alias = value;
  }
  inline GetCoreNetworkPolicyRequest& WithAlias(CoreNetworkPolicyAlias value) {
    SetAlias(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_coreNetworkId;

  int m_policyVersionId{0};

  CoreNetworkPolicyAlias m_alias{CoreNetworkPolicyAlias::NOT_SET};
  bool m_coreNetworkIdHasBeenSet = false;
  bool m_policyVersionIdHasBeenSet = false;
  bool m_aliasHasBeenSet = false;
};

}  // namespace Model
}  // namespace NetworkManager
}  // namespace Aws
