﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/BlockDeviceMapping.h>
#include <aws/ec2/model/CapacityReservationSpecification.h>
#include <aws/ec2/model/CpuOptionsRequest.h>
#include <aws/ec2/model/CreditSpecificationRequest.h>
#include <aws/ec2/model/EnclaveOptionsRequest.h>
#include <aws/ec2/model/HibernationOptionsRequest.h>
#include <aws/ec2/model/IamInstanceProfileSpecification.h>
#include <aws/ec2/model/InstanceIpv6Address.h>
#include <aws/ec2/model/InstanceMaintenanceOptionsRequest.h>
#include <aws/ec2/model/InstanceMarketOptionsRequest.h>
#include <aws/ec2/model/InstanceMetadataOptionsRequest.h>
#include <aws/ec2/model/InstanceNetworkInterfaceSpecification.h>
#include <aws/ec2/model/InstanceNetworkPerformanceOptionsRequest.h>
#include <aws/ec2/model/InstanceSecondaryInterfaceSpecificationRequest.h>
#include <aws/ec2/model/InstanceType.h>
#include <aws/ec2/model/LaunchTemplateSpecification.h>
#include <aws/ec2/model/LicenseConfigurationRequest.h>
#include <aws/ec2/model/OperatorRequest.h>
#include <aws/ec2/model/Placement.h>
#include <aws/ec2/model/PrivateDnsNameOptionsRequest.h>
#include <aws/ec2/model/RunInstancesMonitoringEnabled.h>
#include <aws/ec2/model/ShutdownBehavior.h>
#include <aws/ec2/model/TagSpecification.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class RunInstancesRequest : public EC2Request {
 public:
  AWS_EC2_API RunInstancesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "RunInstances"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The block device mapping, which defines the EBS volumes and instance store
   * volumes to attach to the instance at launch. For more information, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html">Block
   * device mappings</a> in the <i>Amazon EC2 User Guide</i>.</p>
   */
  inline const Aws::Vector<BlockDeviceMapping>& GetBlockDeviceMappings() const { return m_blockDeviceMappings; }
  inline bool BlockDeviceMappingsHasBeenSet() const { return m_blockDeviceMappingsHasBeenSet; }
  template <typename BlockDeviceMappingsT = Aws::Vector<BlockDeviceMapping>>
  void SetBlockDeviceMappings(BlockDeviceMappingsT&& value) {
    m_blockDeviceMappingsHasBeenSet = true;
    m_blockDeviceMappings = std::forward<BlockDeviceMappingsT>(value);
  }
  template <typename BlockDeviceMappingsT = Aws::Vector<BlockDeviceMapping>>
  RunInstancesRequest& WithBlockDeviceMappings(BlockDeviceMappingsT&& value) {
    SetBlockDeviceMappings(std::forward<BlockDeviceMappingsT>(value));
    return *this;
  }
  template <typename BlockDeviceMappingsT = BlockDeviceMapping>
  RunInstancesRequest& AddBlockDeviceMappings(BlockDeviceMappingsT&& value) {
    m_blockDeviceMappingsHasBeenSet = true;
    m_blockDeviceMappings.emplace_back(std::forward<BlockDeviceMappingsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the AMI. An AMI ID is required to launch an instance and must be
   * specified here or in a launch template.</p>
   */
  inline const Aws::String& GetImageId() const { return m_imageId; }
  inline bool ImageIdHasBeenSet() const { return m_imageIdHasBeenSet; }
  template <typename ImageIdT = Aws::String>
  void SetImageId(ImageIdT&& value) {
    m_imageIdHasBeenSet = true;
    m_imageId = std::forward<ImageIdT>(value);
  }
  template <typename ImageIdT = Aws::String>
  RunInstancesRequest& WithImageId(ImageIdT&& value) {
    SetImageId(std::forward<ImageIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The instance type. For more information, see <a
   * href="https://docs.aws.amazon.com/ec2/latest/instancetypes/instance-types.html">Amazon
   * EC2 Instance Types Guide</a>.</p>
   */
  inline InstanceType GetInstanceType() const { return m_instanceType; }
  inline bool InstanceTypeHasBeenSet() const { return m_instanceTypeHasBeenSet; }
  inline void SetInstanceType(InstanceType value) {
    m_instanceTypeHasBeenSet = true;
    m_instanceType = value;
  }
  inline RunInstancesRequest& WithInstanceType(InstanceType value) {
    SetInstanceType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of IPv6 addresses to associate with the primary network interface.
   * Amazon EC2 chooses the IPv6 addresses from the range of your subnet. You cannot
   * specify this option and the option to assign specific IPv6 addresses in the same
   * request. You can specify this option if you've specified a minimum number of
   * instances to launch.</p> <p>You cannot specify this option and the network
   * interfaces option in the same request.</p>
   */
  inline int GetIpv6AddressCount() const { return m_ipv6AddressCount; }
  inline bool Ipv6AddressCountHasBeenSet() const { return m_ipv6AddressCountHasBeenSet; }
  inline void SetIpv6AddressCount(int value) {
    m_ipv6AddressCountHasBeenSet = true;
    m_ipv6AddressCount = value;
  }
  inline RunInstancesRequest& WithIpv6AddressCount(int value) {
    SetIpv6AddressCount(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IPv6 addresses from the range of the subnet to associate with the primary
   * network interface. You cannot specify this option and the option to assign a
   * number of IPv6 addresses in the same request. You cannot specify this option if
   * you've specified a minimum number of instances to launch.</p> <p>You cannot
   * specify this option and the network interfaces option in the same request.</p>
   */
  inline const Aws::Vector<InstanceIpv6Address>& GetIpv6Addresses() const { return m_ipv6Addresses; }
  inline bool Ipv6AddressesHasBeenSet() const { return m_ipv6AddressesHasBeenSet; }
  template <typename Ipv6AddressesT = Aws::Vector<InstanceIpv6Address>>
  void SetIpv6Addresses(Ipv6AddressesT&& value) {
    m_ipv6AddressesHasBeenSet = true;
    m_ipv6Addresses = std::forward<Ipv6AddressesT>(value);
  }
  template <typename Ipv6AddressesT = Aws::Vector<InstanceIpv6Address>>
  RunInstancesRequest& WithIpv6Addresses(Ipv6AddressesT&& value) {
    SetIpv6Addresses(std::forward<Ipv6AddressesT>(value));
    return *this;
  }
  template <typename Ipv6AddressesT = InstanceIpv6Address>
  RunInstancesRequest& AddIpv6Addresses(Ipv6AddressesT&& value) {
    m_ipv6AddressesHasBeenSet = true;
    m_ipv6Addresses.emplace_back(std::forward<Ipv6AddressesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the kernel.</p>  <p>We recommend that you use PV-GRUB
   * instead of kernels and RAM disks. For more information, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedkernels.html">PV-GRUB</a>
   * in the <i>Amazon EC2 User Guide</i>.</p>
   */
  inline const Aws::String& GetKernelId() const { return m_kernelId; }
  inline bool KernelIdHasBeenSet() const { return m_kernelIdHasBeenSet; }
  template <typename KernelIdT = Aws::String>
  void SetKernelId(KernelIdT&& value) {
    m_kernelIdHasBeenSet = true;
    m_kernelId = std::forward<KernelIdT>(value);
  }
  template <typename KernelIdT = Aws::String>
  RunInstancesRequest& WithKernelId(KernelIdT&& value) {
    SetKernelId(std::forward<KernelIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the key pair. For more information, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/create-key-pairs.html">Create
   * a key pair for your EC2 instance</a>.</p>  <p>If you do not specify a
   * key pair, you can't connect to the instance unless you choose an AMI that is
   * configured to allow users another way to log in.</p>
   */
  inline const Aws::String& GetKeyName() const { return m_keyName; }
  inline bool KeyNameHasBeenSet() const { return m_keyNameHasBeenSet; }
  template <typename KeyNameT = Aws::String>
  void SetKeyName(KeyNameT&& value) {
    m_keyNameHasBeenSet = true;
    m_keyName = std::forward<KeyNameT>(value);
  }
  template <typename KeyNameT = Aws::String>
  RunInstancesRequest& WithKeyName(KeyNameT&& value) {
    SetKeyName(std::forward<KeyNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of instances to launch. If you specify a value that is
   * more capacity than Amazon EC2 can launch in the target Availability Zone, Amazon
   * EC2 launches the largest possible number of instances above the specified
   * minimum count.</p> <p>Constraints: Between 1 and the quota for the specified
   * instance type for your account for this Region. For more information, see <a
   * href="https://docs.aws.amazon.com/ec2/latest/instancetypes/ec2-instance-quotas.html">Amazon
   * EC2 instance type quotas</a>.</p>
   */
  inline int GetMaxCount() const { return m_maxCount; }
  inline bool MaxCountHasBeenSet() const { return m_maxCountHasBeenSet; }
  inline void SetMaxCount(int value) {
    m_maxCountHasBeenSet = true;
    m_maxCount = value;
  }
  inline RunInstancesRequest& WithMaxCount(int value) {
    SetMaxCount(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The minimum number of instances to launch. If you specify a value that is
   * more capacity than Amazon EC2 can provide in the target Availability Zone,
   * Amazon EC2 does not launch any instances.</p> <p>Constraints: Between 1 and the
   * quota for the specified instance type for your account for this Region. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/ec2/latest/instancetypes/ec2-instance-quotas.html">Amazon
   * EC2 instance type quotas</a>.</p>
   */
  inline int GetMinCount() const { return m_minCount; }
  inline bool MinCountHasBeenSet() const { return m_minCountHasBeenSet; }
  inline void SetMinCount(int value) {
    m_minCountHasBeenSet = true;
    m_minCount = value;
  }
  inline RunInstancesRequest& WithMinCount(int value) {
    SetMinCount(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether detailed monitoring is enabled for the instance.</p>
   */
  inline const RunInstancesMonitoringEnabled& GetMonitoring() const { return m_monitoring; }
  inline bool MonitoringHasBeenSet() const { return m_monitoringHasBeenSet; }
  template <typename MonitoringT = RunInstancesMonitoringEnabled>
  void SetMonitoring(MonitoringT&& value) {
    m_monitoringHasBeenSet = true;
    m_monitoring = std::forward<MonitoringT>(value);
  }
  template <typename MonitoringT = RunInstancesMonitoringEnabled>
  RunInstancesRequest& WithMonitoring(MonitoringT&& value) {
    SetMonitoring(std::forward<MonitoringT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The placement for the instance.</p>
   */
  inline const Placement& GetPlacement() const { return m_placement; }
  inline bool PlacementHasBeenSet() const { return m_placementHasBeenSet; }
  template <typename PlacementT = Placement>
  void SetPlacement(PlacementT&& value) {
    m_placementHasBeenSet = true;
    m_placement = std::forward<PlacementT>(value);
  }
  template <typename PlacementT = Placement>
  RunInstancesRequest& WithPlacement(PlacementT&& value) {
    SetPlacement(std::forward<PlacementT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the RAM disk to select. Some kernels require additional drivers at
   * launch. Check the kernel requirements for information about whether you need to
   * specify a RAM disk. To find kernel requirements, go to the Amazon Web Services
   * Resource Center and search for the kernel ID.</p>  <p>We recommend
   * that you use PV-GRUB instead of kernels and RAM disks. For more information, see
   * <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedkernels.html">PV-GRUB</a>
   * in the <i>Amazon EC2 User Guide</i>.</p>
   */
  inline const Aws::String& GetRamdiskId() const { return m_ramdiskId; }
  inline bool RamdiskIdHasBeenSet() const { return m_ramdiskIdHasBeenSet; }
  template <typename RamdiskIdT = Aws::String>
  void SetRamdiskId(RamdiskIdT&& value) {
    m_ramdiskIdHasBeenSet = true;
    m_ramdiskId = std::forward<RamdiskIdT>(value);
  }
  template <typename RamdiskIdT = Aws::String>
  RunInstancesRequest& WithRamdiskId(RamdiskIdT&& value) {
    SetRamdiskId(std::forward<RamdiskIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IDs of the security groups.</p> <p>If you specify a network interface,
   * you must specify any security groups as part of the network interface instead of
   * using this parameter.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSecurityGroupIds() const { return m_securityGroupIds; }
  inline bool SecurityGroupIdsHasBeenSet() const { return m_securityGroupIdsHasBeenSet; }
  template <typename SecurityGroupIdsT = Aws::Vector<Aws::String>>
  void SetSecurityGroupIds(SecurityGroupIdsT&& value) {
    m_securityGroupIdsHasBeenSet = true;
    m_securityGroupIds = std::forward<SecurityGroupIdsT>(value);
  }
  template <typename SecurityGroupIdsT = Aws::Vector<Aws::String>>
  RunInstancesRequest& WithSecurityGroupIds(SecurityGroupIdsT&& value) {
    SetSecurityGroupIds(std::forward<SecurityGroupIdsT>(value));
    return *this;
  }
  template <typename SecurityGroupIdsT = Aws::String>
  RunInstancesRequest& AddSecurityGroupIds(SecurityGroupIdsT&& value) {
    m_securityGroupIdsHasBeenSet = true;
    m_securityGroupIds.emplace_back(std::forward<SecurityGroupIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>[Default VPC] The names of the security groups.</p> <p>If you specify a
   * network interface, you must specify any security groups as part of the network
   * interface instead of using this parameter.</p> <p>Default: Amazon EC2 uses the
   * default security group.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSecurityGroups() const { return m_securityGroups; }
  inline bool SecurityGroupsHasBeenSet() const { return m_securityGroupsHasBeenSet; }
  template <typename SecurityGroupsT = Aws::Vector<Aws::String>>
  void SetSecurityGroups(SecurityGroupsT&& value) {
    m_securityGroupsHasBeenSet = true;
    m_securityGroups = std::forward<SecurityGroupsT>(value);
  }
  template <typename SecurityGroupsT = Aws::Vector<Aws::String>>
  RunInstancesRequest& WithSecurityGroups(SecurityGroupsT&& value) {
    SetSecurityGroups(std::forward<SecurityGroupsT>(value));
    return *this;
  }
  template <typename SecurityGroupsT = Aws::String>
  RunInstancesRequest& AddSecurityGroups(SecurityGroupsT&& value) {
    m_securityGroupsHasBeenSet = true;
    m_securityGroups.emplace_back(std::forward<SecurityGroupsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the subnet to launch the instance into.</p> <p>If you specify a
   * network interface, you must specify any subnets as part of the network interface
   * instead of using this parameter.</p>
   */
  inline const Aws::String& GetSubnetId() const { return m_subnetId; }
  inline bool SubnetIdHasBeenSet() const { return m_subnetIdHasBeenSet; }
  template <typename SubnetIdT = Aws::String>
  void SetSubnetId(SubnetIdT&& value) {
    m_subnetIdHasBeenSet = true;
    m_subnetId = std::forward<SubnetIdT>(value);
  }
  template <typename SubnetIdT = Aws::String>
  RunInstancesRequest& WithSubnetId(SubnetIdT&& value) {
    SetSubnetId(std::forward<SubnetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The user data to make available to the instance. User data must be
   * base64-encoded. Depending on the tool or SDK that you're using, the
   * base64-encoding might be performed for you. For more information, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/user-data.html">Run
   * commands at launch using instance user data</a>.</p>
   */
  inline const Aws::String& GetUserData() const { return m_userData; }
  inline bool UserDataHasBeenSet() const { return m_userDataHasBeenSet; }
  template <typename UserDataT = Aws::String>
  void SetUserData(UserDataT&& value) {
    m_userDataHasBeenSet = true;
    m_userData = std::forward<UserDataT>(value);
  }
  template <typename UserDataT = Aws::String>
  RunInstancesRequest& WithUserData(UserDataT&& value) {
    SetUserData(std::forward<UserDataT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags to apply to the resources that are created during instance
   * launch.</p> <p>You can specify tags for the following resources only:</p> <ul>
   * <li> <p>Instances</p> </li> <li> <p>Volumes</p> </li> <li> <p>Spot Instance
   * requests</p> </li> <li> <p>Network interfaces</p> </li> </ul> <p>To tag a
   * resource after it has been created, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html">CreateTags</a>.</p>
   */
  inline const Aws::Vector<TagSpecification>& GetTagSpecifications() const { return m_tagSpecifications; }
  inline bool TagSpecificationsHasBeenSet() const { return m_tagSpecificationsHasBeenSet; }
  template <typename TagSpecificationsT = Aws::Vector<TagSpecification>>
  void SetTagSpecifications(TagSpecificationsT&& value) {
    m_tagSpecificationsHasBeenSet = true;
    m_tagSpecifications = std::forward<TagSpecificationsT>(value);
  }
  template <typename TagSpecificationsT = Aws::Vector<TagSpecification>>
  RunInstancesRequest& WithTagSpecifications(TagSpecificationsT&& value) {
    SetTagSpecifications(std::forward<TagSpecificationsT>(value));
    return *this;
  }
  template <typename TagSpecificationsT = TagSpecification>
  RunInstancesRequest& AddTagSpecifications(TagSpecificationsT&& value) {
    m_tagSpecificationsHasBeenSet = true;
    m_tagSpecifications.emplace_back(std::forward<TagSpecificationsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The launch template. Any additional parameters that you specify for the new
   * instance overwrite the corresponding parameters included in the launch
   * template.</p>
   */
  inline const LaunchTemplateSpecification& GetLaunchTemplate() const { return m_launchTemplate; }
  inline bool LaunchTemplateHasBeenSet() const { return m_launchTemplateHasBeenSet; }
  template <typename LaunchTemplateT = LaunchTemplateSpecification>
  void SetLaunchTemplate(LaunchTemplateT&& value) {
    m_launchTemplateHasBeenSet = true;
    m_launchTemplate = std::forward<LaunchTemplateT>(value);
  }
  template <typename LaunchTemplateT = LaunchTemplateSpecification>
  RunInstancesRequest& WithLaunchTemplate(LaunchTemplateT&& value) {
    SetLaunchTemplate(std::forward<LaunchTemplateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The market (purchasing) option for the instances.</p> <p>For
   * <a>RunInstances</a>, persistent Spot Instance requests are only supported when
   * <b>InstanceInterruptionBehavior</b> is set to either <code>hibernate</code> or
   * <code>stop</code>.</p>
   */
  inline const InstanceMarketOptionsRequest& GetInstanceMarketOptions() const { return m_instanceMarketOptions; }
  inline bool InstanceMarketOptionsHasBeenSet() const { return m_instanceMarketOptionsHasBeenSet; }
  template <typename InstanceMarketOptionsT = InstanceMarketOptionsRequest>
  void SetInstanceMarketOptions(InstanceMarketOptionsT&& value) {
    m_instanceMarketOptionsHasBeenSet = true;
    m_instanceMarketOptions = std::forward<InstanceMarketOptionsT>(value);
  }
  template <typename InstanceMarketOptionsT = InstanceMarketOptionsRequest>
  RunInstancesRequest& WithInstanceMarketOptions(InstanceMarketOptionsT&& value) {
    SetInstanceMarketOptions(std::forward<InstanceMarketOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The credit option for CPU usage of the burstable performance instance. Valid
   * values are <code>standard</code> and <code>unlimited</code>. To change this
   * attribute after launch, use <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCreditSpecification.html">
   * ModifyInstanceCreditSpecification</a>. For more information, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html">Burstable
   * performance instances</a> in the <i>Amazon EC2 User Guide</i>.</p> <p>Default:
   * <code>standard</code> (T2 instances) or <code>unlimited</code> (T3/T3a/T4g
   * instances)</p> <p>For T3 instances with <code>host</code> tenancy, only
   * <code>standard</code> is supported.</p>
   */
  inline const CreditSpecificationRequest& GetCreditSpecification() const { return m_creditSpecification; }
  inline bool CreditSpecificationHasBeenSet() const { return m_creditSpecificationHasBeenSet; }
  template <typename CreditSpecificationT = CreditSpecificationRequest>
  void SetCreditSpecification(CreditSpecificationT&& value) {
    m_creditSpecificationHasBeenSet = true;
    m_creditSpecification = std::forward<CreditSpecificationT>(value);
  }
  template <typename CreditSpecificationT = CreditSpecificationRequest>
  RunInstancesRequest& WithCreditSpecification(CreditSpecificationT&& value) {
    SetCreditSpecification(std::forward<CreditSpecificationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The CPU options for the instance. For more information, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-optimize-cpu.html">Optimize
   * CPU options</a> in the <i>Amazon EC2 User Guide</i>.</p>
   */
  inline const CpuOptionsRequest& GetCpuOptions() const { return m_cpuOptions; }
  inline bool CpuOptionsHasBeenSet() const { return m_cpuOptionsHasBeenSet; }
  template <typename CpuOptionsT = CpuOptionsRequest>
  void SetCpuOptions(CpuOptionsT&& value) {
    m_cpuOptionsHasBeenSet = true;
    m_cpuOptions = std::forward<CpuOptionsT>(value);
  }
  template <typename CpuOptionsT = CpuOptionsRequest>
  RunInstancesRequest& WithCpuOptions(CpuOptionsT&& value) {
    SetCpuOptions(std::forward<CpuOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about the Capacity Reservation targeting option. If you do not
   * specify this parameter, the instance's Capacity Reservation preference defaults
   * to <code>open</code>, which enables it to run in any open Capacity Reservation
   * that has matching attributes (instance type, platform, Availability Zone, and
   * tenancy).</p>
   */
  inline const CapacityReservationSpecification& GetCapacityReservationSpecification() const { return m_capacityReservationSpecification; }
  inline bool CapacityReservationSpecificationHasBeenSet() const { return m_capacityReservationSpecificationHasBeenSet; }
  template <typename CapacityReservationSpecificationT = CapacityReservationSpecification>
  void SetCapacityReservationSpecification(CapacityReservationSpecificationT&& value) {
    m_capacityReservationSpecificationHasBeenSet = true;
    m_capacityReservationSpecification = std::forward<CapacityReservationSpecificationT>(value);
  }
  template <typename CapacityReservationSpecificationT = CapacityReservationSpecification>
  RunInstancesRequest& WithCapacityReservationSpecification(CapacityReservationSpecificationT&& value) {
    SetCapacityReservationSpecification(std::forward<CapacityReservationSpecificationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether an instance is enabled for hibernation. This parameter is
   * valid only if the instance meets the <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/hibernating-prerequisites.html">hibernation
   * prerequisites</a>. For more information, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Hibernate.html">Hibernate
   * your Amazon EC2 instance</a> in the <i>Amazon EC2 User Guide</i>.</p> <p>You
   * can't enable hibernation and Amazon Web Services Nitro Enclaves on the same
   * instance.</p>
   */
  inline const HibernationOptionsRequest& GetHibernationOptions() const { return m_hibernationOptions; }
  inline bool HibernationOptionsHasBeenSet() const { return m_hibernationOptionsHasBeenSet; }
  template <typename HibernationOptionsT = HibernationOptionsRequest>
  void SetHibernationOptions(HibernationOptionsT&& value) {
    m_hibernationOptionsHasBeenSet = true;
    m_hibernationOptions = std::forward<HibernationOptionsT>(value);
  }
  template <typename HibernationOptionsT = HibernationOptionsRequest>
  RunInstancesRequest& WithHibernationOptions(HibernationOptionsT&& value) {
    SetHibernationOptions(std::forward<HibernationOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The license configurations.</p>
   */
  inline const Aws::Vector<LicenseConfigurationRequest>& GetLicenseSpecifications() const { return m_licenseSpecifications; }
  inline bool LicenseSpecificationsHasBeenSet() const { return m_licenseSpecificationsHasBeenSet; }
  template <typename LicenseSpecificationsT = Aws::Vector<LicenseConfigurationRequest>>
  void SetLicenseSpecifications(LicenseSpecificationsT&& value) {
    m_licenseSpecificationsHasBeenSet = true;
    m_licenseSpecifications = std::forward<LicenseSpecificationsT>(value);
  }
  template <typename LicenseSpecificationsT = Aws::Vector<LicenseConfigurationRequest>>
  RunInstancesRequest& WithLicenseSpecifications(LicenseSpecificationsT&& value) {
    SetLicenseSpecifications(std::forward<LicenseSpecificationsT>(value));
    return *this;
  }
  template <typename LicenseSpecificationsT = LicenseConfigurationRequest>
  RunInstancesRequest& AddLicenseSpecifications(LicenseSpecificationsT&& value) {
    m_licenseSpecificationsHasBeenSet = true;
    m_licenseSpecifications.emplace_back(std::forward<LicenseSpecificationsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The metadata options for the instance. For more information, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/configuring-instance-metadata-options.html">Configure
   * the Instance Metadata Service options</a>.</p>
   */
  inline const InstanceMetadataOptionsRequest& GetMetadataOptions() const { return m_metadataOptions; }
  inline bool MetadataOptionsHasBeenSet() const { return m_metadataOptionsHasBeenSet; }
  template <typename MetadataOptionsT = InstanceMetadataOptionsRequest>
  void SetMetadataOptions(MetadataOptionsT&& value) {
    m_metadataOptionsHasBeenSet = true;
    m_metadataOptions = std::forward<MetadataOptionsT>(value);
  }
  template <typename MetadataOptionsT = InstanceMetadataOptionsRequest>
  RunInstancesRequest& WithMetadataOptions(MetadataOptionsT&& value) {
    SetMetadataOptions(std::forward<MetadataOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether the instance is enabled for Amazon Web Services Nitro
   * Enclaves. For more information, see <a
   * href="https://docs.aws.amazon.com/enclaves/latest/user/">Amazon Web Services
   * Nitro Enclaves User Guide</a>.</p> <p>You can't enable Amazon Web Services Nitro
   * Enclaves and hibernation on the same instance.</p>
   */
  inline const EnclaveOptionsRequest& GetEnclaveOptions() const { return m_enclaveOptions; }
  inline bool EnclaveOptionsHasBeenSet() const { return m_enclaveOptionsHasBeenSet; }
  template <typename EnclaveOptionsT = EnclaveOptionsRequest>
  void SetEnclaveOptions(EnclaveOptionsT&& value) {
    m_enclaveOptionsHasBeenSet = true;
    m_enclaveOptions = std::forward<EnclaveOptionsT>(value);
  }
  template <typename EnclaveOptionsT = EnclaveOptionsRequest>
  RunInstancesRequest& WithEnclaveOptions(EnclaveOptionsT&& value) {
    SetEnclaveOptions(std::forward<EnclaveOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The options for the instance hostname. The default values are inherited from
   * the subnet. Applies only if creating a network interface, not attaching an
   * existing one.</p>
   */
  inline const PrivateDnsNameOptionsRequest& GetPrivateDnsNameOptions() const { return m_privateDnsNameOptions; }
  inline bool PrivateDnsNameOptionsHasBeenSet() const { return m_privateDnsNameOptionsHasBeenSet; }
  template <typename PrivateDnsNameOptionsT = PrivateDnsNameOptionsRequest>
  void SetPrivateDnsNameOptions(PrivateDnsNameOptionsT&& value) {
    m_privateDnsNameOptionsHasBeenSet = true;
    m_privateDnsNameOptions = std::forward<PrivateDnsNameOptionsT>(value);
  }
  template <typename PrivateDnsNameOptionsT = PrivateDnsNameOptionsRequest>
  RunInstancesRequest& WithPrivateDnsNameOptions(PrivateDnsNameOptionsT&& value) {
    SetPrivateDnsNameOptions(std::forward<PrivateDnsNameOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maintenance and recovery options for the instance.</p>
   */
  inline const InstanceMaintenanceOptionsRequest& GetMaintenanceOptions() const { return m_maintenanceOptions; }
  inline bool MaintenanceOptionsHasBeenSet() const { return m_maintenanceOptionsHasBeenSet; }
  template <typename MaintenanceOptionsT = InstanceMaintenanceOptionsRequest>
  void SetMaintenanceOptions(MaintenanceOptionsT&& value) {
    m_maintenanceOptionsHasBeenSet = true;
    m_maintenanceOptions = std::forward<MaintenanceOptionsT>(value);
  }
  template <typename MaintenanceOptionsT = InstanceMaintenanceOptionsRequest>
  RunInstancesRequest& WithMaintenanceOptions(MaintenanceOptionsT&& value) {
    SetMaintenanceOptions(std::forward<MaintenanceOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether an instance is enabled for stop protection. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-stop-protection.html">Enable
   * stop protection for your EC2 instances</a>.</p>
   */
  inline bool GetDisableApiStop() const { return m_disableApiStop; }
  inline bool DisableApiStopHasBeenSet() const { return m_disableApiStopHasBeenSet; }
  inline void SetDisableApiStop(bool value) {
    m_disableApiStopHasBeenSet = true;
    m_disableApiStop = value;
  }
  inline RunInstancesRequest& WithDisableApiStop(bool value) {
    SetDisableApiStop(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If you’re launching an instance into a dual-stack or IPv6-only subnet, you
   * can enable assigning a primary IPv6 address. A primary IPv6 address is an IPv6
   * GUA address associated with an ENI that you have enabled to use a primary IPv6
   * address. Use this option if an instance relies on its IPv6 address not changing.
   * When you launch the instance, Amazon Web Services will automatically assign an
   * IPv6 address associated with the ENI attached to your instance to be the primary
   * IPv6 address. Once you enable an IPv6 GUA address to be a primary IPv6, you
   * cannot disable it. When you enable an IPv6 GUA address to be a primary IPv6, the
   * first IPv6 GUA will be made the primary IPv6 address until the instance is
   * terminated or the network interface is detached. If you have multiple IPv6
   * addresses associated with an ENI attached to your instance and you enable a
   * primary IPv6 address, the first IPv6 GUA address associated with the ENI becomes
   * the primary IPv6 address.</p>
   */
  inline bool GetEnablePrimaryIpv6() const { return m_enablePrimaryIpv6; }
  inline bool EnablePrimaryIpv6HasBeenSet() const { return m_enablePrimaryIpv6HasBeenSet; }
  inline void SetEnablePrimaryIpv6(bool value) {
    m_enablePrimaryIpv6HasBeenSet = true;
    m_enablePrimaryIpv6 = value;
  }
  inline RunInstancesRequest& WithEnablePrimaryIpv6(bool value) {
    SetEnablePrimaryIpv6(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Contains settings for the network performance options for the instance.</p>
   */
  inline const InstanceNetworkPerformanceOptionsRequest& GetNetworkPerformanceOptions() const { return m_networkPerformanceOptions; }
  inline bool NetworkPerformanceOptionsHasBeenSet() const { return m_networkPerformanceOptionsHasBeenSet; }
  template <typename NetworkPerformanceOptionsT = InstanceNetworkPerformanceOptionsRequest>
  void SetNetworkPerformanceOptions(NetworkPerformanceOptionsT&& value) {
    m_networkPerformanceOptionsHasBeenSet = true;
    m_networkPerformanceOptions = std::forward<NetworkPerformanceOptionsT>(value);
  }
  template <typename NetworkPerformanceOptionsT = InstanceNetworkPerformanceOptionsRequest>
  RunInstancesRequest& WithNetworkPerformanceOptions(NetworkPerformanceOptionsT&& value) {
    SetNetworkPerformanceOptions(std::forward<NetworkPerformanceOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Reserved for internal use.</p>
   */
  inline const OperatorRequest& GetOperator() const { return m_operator; }
  inline bool OperatorHasBeenSet() const { return m_operatorHasBeenSet; }
  template <typename OperatorT = OperatorRequest>
  void SetOperator(OperatorT&& value) {
    m_operatorHasBeenSet = true;
    m_operator = std::forward<OperatorT>(value);
  }
  template <typename OperatorT = OperatorRequest>
  RunInstancesRequest& WithOperator(OperatorT&& value) {
    SetOperator(std::forward<OperatorT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The secondary interfaces to associate with the instance.</p>
   */
  inline const Aws::Vector<InstanceSecondaryInterfaceSpecificationRequest>& GetSecondaryInterfaces() const { return m_secondaryInterfaces; }
  inline bool SecondaryInterfacesHasBeenSet() const { return m_secondaryInterfacesHasBeenSet; }
  template <typename SecondaryInterfacesT = Aws::Vector<InstanceSecondaryInterfaceSpecificationRequest>>
  void SetSecondaryInterfaces(SecondaryInterfacesT&& value) {
    m_secondaryInterfacesHasBeenSet = true;
    m_secondaryInterfaces = std::forward<SecondaryInterfacesT>(value);
  }
  template <typename SecondaryInterfacesT = Aws::Vector<InstanceSecondaryInterfaceSpecificationRequest>>
  RunInstancesRequest& WithSecondaryInterfaces(SecondaryInterfacesT&& value) {
    SetSecondaryInterfaces(std::forward<SecondaryInterfacesT>(value));
    return *this;
  }
  template <typename SecondaryInterfacesT = InstanceSecondaryInterfaceSpecificationRequest>
  RunInstancesRequest& AddSecondaryInterfaces(SecondaryInterfacesT&& value) {
    m_secondaryInterfacesHasBeenSet = true;
    m_secondaryInterfaces.emplace_back(std::forward<SecondaryInterfacesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Checks whether you have the required permissions for the operation, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline RunInstancesRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether termination protection is enabled for the instance. The
   * default is <code>false</code>, which means that you can terminate the instance
   * using the Amazon EC2 console, command line tools, or API. You can enable
   * termination protection when you launch an instance, while the instance is
   * running, or while the instance is stopped.</p>
   */
  inline bool GetDisableApiTermination() const { return m_disableApiTermination; }
  inline bool DisableApiTerminationHasBeenSet() const { return m_disableApiTerminationHasBeenSet; }
  inline void SetDisableApiTermination(bool value) {
    m_disableApiTerminationHasBeenSet = true;
    m_disableApiTermination = value;
  }
  inline RunInstancesRequest& WithDisableApiTermination(bool value) {
    SetDisableApiTermination(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether an instance stops or terminates when you initiate shutdown
   * from the instance (using the operating system command for system shutdown).</p>
   * <p>Default: <code>stop</code> </p>
   */
  inline ShutdownBehavior GetInstanceInitiatedShutdownBehavior() const { return m_instanceInitiatedShutdownBehavior; }
  inline bool InstanceInitiatedShutdownBehaviorHasBeenSet() const { return m_instanceInitiatedShutdownBehaviorHasBeenSet; }
  inline void SetInstanceInitiatedShutdownBehavior(ShutdownBehavior value) {
    m_instanceInitiatedShutdownBehaviorHasBeenSet = true;
    m_instanceInitiatedShutdownBehavior = value;
  }
  inline RunInstancesRequest& WithInstanceInitiatedShutdownBehavior(ShutdownBehavior value) {
    SetInstanceInitiatedShutdownBehavior(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The primary IPv4 address. You must specify a value from the IPv4 address
   * range of the subnet.</p> <p>Only one private IP address can be designated as
   * primary. You can't specify this option if you've specified the option to
   * designate a private IP address as the primary IP address in a network interface
   * specification. You cannot specify this option if you're launching more than one
   * instance in the request.</p> <p>You cannot specify this option and the network
   * interfaces option in the same request.</p>
   */
  inline const Aws::String& GetPrivateIpAddress() const { return m_privateIpAddress; }
  inline bool PrivateIpAddressHasBeenSet() const { return m_privateIpAddressHasBeenSet; }
  template <typename PrivateIpAddressT = Aws::String>
  void SetPrivateIpAddress(PrivateIpAddressT&& value) {
    m_privateIpAddressHasBeenSet = true;
    m_privateIpAddress = std::forward<PrivateIpAddressT>(value);
  }
  template <typename PrivateIpAddressT = Aws::String>
  RunInstancesRequest& WithPrivateIpAddress(PrivateIpAddressT&& value) {
    SetPrivateIpAddress(std::forward<PrivateIpAddressT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Unique, case-sensitive identifier you provide to ensure the idempotency of
   * the request. If you do not specify a client token, a randomly generated token is
   * used for the request to ensure idempotency.</p> <p>For more information, see <a
   * href="https://docs.aws.amazon.com/ec2/latest/devguide/ec2-api-idempotency.html">Ensuring
   * idempotency in Amazon EC2 API requests</a>.</p> <p>Constraints: Maximum 64 ASCII
   * characters</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  RunInstancesRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Reserved.</p>
   */
  inline const Aws::String& GetAdditionalInfo() const { return m_additionalInfo; }
  inline bool AdditionalInfoHasBeenSet() const { return m_additionalInfoHasBeenSet; }
  template <typename AdditionalInfoT = Aws::String>
  void SetAdditionalInfo(AdditionalInfoT&& value) {
    m_additionalInfoHasBeenSet = true;
    m_additionalInfo = std::forward<AdditionalInfoT>(value);
  }
  template <typename AdditionalInfoT = Aws::String>
  RunInstancesRequest& WithAdditionalInfo(AdditionalInfoT&& value) {
    SetAdditionalInfo(std::forward<AdditionalInfoT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The network interfaces to associate with the instance.</p>
   */
  inline const Aws::Vector<InstanceNetworkInterfaceSpecification>& GetNetworkInterfaces() const { return m_networkInterfaces; }
  inline bool NetworkInterfacesHasBeenSet() const { return m_networkInterfacesHasBeenSet; }
  template <typename NetworkInterfacesT = Aws::Vector<InstanceNetworkInterfaceSpecification>>
  void SetNetworkInterfaces(NetworkInterfacesT&& value) {
    m_networkInterfacesHasBeenSet = true;
    m_networkInterfaces = std::forward<NetworkInterfacesT>(value);
  }
  template <typename NetworkInterfacesT = Aws::Vector<InstanceNetworkInterfaceSpecification>>
  RunInstancesRequest& WithNetworkInterfaces(NetworkInterfacesT&& value) {
    SetNetworkInterfaces(std::forward<NetworkInterfacesT>(value));
    return *this;
  }
  template <typename NetworkInterfacesT = InstanceNetworkInterfaceSpecification>
  RunInstancesRequest& AddNetworkInterfaces(NetworkInterfacesT&& value) {
    m_networkInterfacesHasBeenSet = true;
    m_networkInterfaces.emplace_back(std::forward<NetworkInterfacesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name or Amazon Resource Name (ARN) of an IAM instance profile.</p>
   */
  inline const IamInstanceProfileSpecification& GetIamInstanceProfile() const { return m_iamInstanceProfile; }
  inline bool IamInstanceProfileHasBeenSet() const { return m_iamInstanceProfileHasBeenSet; }
  template <typename IamInstanceProfileT = IamInstanceProfileSpecification>
  void SetIamInstanceProfile(IamInstanceProfileT&& value) {
    m_iamInstanceProfileHasBeenSet = true;
    m_iamInstanceProfile = std::forward<IamInstanceProfileT>(value);
  }
  template <typename IamInstanceProfileT = IamInstanceProfileSpecification>
  RunInstancesRequest& WithIamInstanceProfile(IamInstanceProfileT&& value) {
    SetIamInstanceProfile(std::forward<IamInstanceProfileT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether the instance is optimized for Amazon EBS I/O. This
   * optimization provides dedicated throughput to Amazon EBS and an optimized
   * configuration stack to provide optimal Amazon EBS I/O performance. This
   * optimization isn't available with all instance types. Additional usage charges
   * apply when using an EBS-optimized instance.</p> <p>Default: <code>false</code>
   * </p>
   */
  inline bool GetEbsOptimized() const { return m_ebsOptimized; }
  inline bool EbsOptimizedHasBeenSet() const { return m_ebsOptimizedHasBeenSet; }
  inline void SetEbsOptimized(bool value) {
    m_ebsOptimizedHasBeenSet = true;
    m_ebsOptimized = value;
  }
  inline RunInstancesRequest& WithEbsOptimized(bool value) {
    SetEbsOptimized(value);
    return *this;
  }
  ///@}
 private:
  Aws::Vector<BlockDeviceMapping> m_blockDeviceMappings;

  Aws::String m_imageId;

  InstanceType m_instanceType{InstanceType::NOT_SET};

  int m_ipv6AddressCount{0};

  Aws::Vector<InstanceIpv6Address> m_ipv6Addresses;

  Aws::String m_kernelId;

  Aws::String m_keyName;

  int m_maxCount{0};

  int m_minCount{0};

  RunInstancesMonitoringEnabled m_monitoring;

  Placement m_placement;

  Aws::String m_ramdiskId;

  Aws::Vector<Aws::String> m_securityGroupIds;

  Aws::Vector<Aws::String> m_securityGroups;

  Aws::String m_subnetId;

  Aws::String m_userData;

  Aws::Vector<TagSpecification> m_tagSpecifications;

  LaunchTemplateSpecification m_launchTemplate;

  InstanceMarketOptionsRequest m_instanceMarketOptions;

  CreditSpecificationRequest m_creditSpecification;

  CpuOptionsRequest m_cpuOptions;

  CapacityReservationSpecification m_capacityReservationSpecification;

  HibernationOptionsRequest m_hibernationOptions;

  Aws::Vector<LicenseConfigurationRequest> m_licenseSpecifications;

  InstanceMetadataOptionsRequest m_metadataOptions;

  EnclaveOptionsRequest m_enclaveOptions;

  PrivateDnsNameOptionsRequest m_privateDnsNameOptions;

  InstanceMaintenanceOptionsRequest m_maintenanceOptions;

  bool m_disableApiStop{false};

  bool m_enablePrimaryIpv6{false};

  InstanceNetworkPerformanceOptionsRequest m_networkPerformanceOptions;

  OperatorRequest m_operator;

  Aws::Vector<InstanceSecondaryInterfaceSpecificationRequest> m_secondaryInterfaces;

  bool m_dryRun{false};

  bool m_disableApiTermination{false};

  ShutdownBehavior m_instanceInitiatedShutdownBehavior{ShutdownBehavior::NOT_SET};

  Aws::String m_privateIpAddress;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};

  Aws::String m_additionalInfo;

  Aws::Vector<InstanceNetworkInterfaceSpecification> m_networkInterfaces;

  IamInstanceProfileSpecification m_iamInstanceProfile;

  bool m_ebsOptimized{false};
  bool m_blockDeviceMappingsHasBeenSet = false;
  bool m_imageIdHasBeenSet = false;
  bool m_instanceTypeHasBeenSet = false;
  bool m_ipv6AddressCountHasBeenSet = false;
  bool m_ipv6AddressesHasBeenSet = false;
  bool m_kernelIdHasBeenSet = false;
  bool m_keyNameHasBeenSet = false;
  bool m_maxCountHasBeenSet = false;
  bool m_minCountHasBeenSet = false;
  bool m_monitoringHasBeenSet = false;
  bool m_placementHasBeenSet = false;
  bool m_ramdiskIdHasBeenSet = false;
  bool m_securityGroupIdsHasBeenSet = false;
  bool m_securityGroupsHasBeenSet = false;
  bool m_subnetIdHasBeenSet = false;
  bool m_userDataHasBeenSet = false;
  bool m_tagSpecificationsHasBeenSet = false;
  bool m_launchTemplateHasBeenSet = false;
  bool m_instanceMarketOptionsHasBeenSet = false;
  bool m_creditSpecificationHasBeenSet = false;
  bool m_cpuOptionsHasBeenSet = false;
  bool m_capacityReservationSpecificationHasBeenSet = false;
  bool m_hibernationOptionsHasBeenSet = false;
  bool m_licenseSpecificationsHasBeenSet = false;
  bool m_metadataOptionsHasBeenSet = false;
  bool m_enclaveOptionsHasBeenSet = false;
  bool m_privateDnsNameOptionsHasBeenSet = false;
  bool m_maintenanceOptionsHasBeenSet = false;
  bool m_disableApiStopHasBeenSet = false;
  bool m_enablePrimaryIpv6HasBeenSet = false;
  bool m_networkPerformanceOptionsHasBeenSet = false;
  bool m_operatorHasBeenSet = false;
  bool m_secondaryInterfacesHasBeenSet = false;
  bool m_dryRunHasBeenSet = false;
  bool m_disableApiTerminationHasBeenSet = false;
  bool m_instanceInitiatedShutdownBehaviorHasBeenSet = false;
  bool m_privateIpAddressHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
  bool m_additionalInfoHasBeenSet = false;
  bool m_networkInterfacesHasBeenSet = false;
  bool m_iamInstanceProfileHasBeenSet = false;
  bool m_ebsOptimizedHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
