/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.credential;

import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Meter;
import io.micrometer.core.instrument.Tag;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.jboss.logging.Logger;
import org.keycloak.common.util.Time;
import org.keycloak.credential.CredentialInput;
import org.keycloak.credential.CredentialInputUpdater;
import org.keycloak.credential.CredentialInputValidator;
import org.keycloak.credential.CredentialModel;
import org.keycloak.credential.CredentialProvider;
import org.keycloak.credential.CredentialTypeMetadata;
import org.keycloak.credential.CredentialTypeMetadataContext;
import org.keycloak.credential.hash.PasswordHashProvider;
import org.keycloak.models.AbstractKeycloakTransaction;
import org.keycloak.models.KeycloakContext;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.KeycloakSessionFactory;
import org.keycloak.models.KeycloakTransaction;
import org.keycloak.models.ModelException;
import org.keycloak.models.PasswordPolicy;
import org.keycloak.models.RealmModel;
import org.keycloak.models.UserCredentialModel;
import org.keycloak.models.UserModel;
import org.keycloak.models.credential.PasswordCredentialModel;
import org.keycloak.models.utils.KeycloakModelUtils;
import org.keycloak.policy.PasswordPolicyManagerProvider;
import org.keycloak.policy.PolicyError;

public class PasswordCredentialProvider
implements CredentialProvider<PasswordCredentialModel>,
CredentialInputUpdater,
CredentialInputValidator {
    private static final Logger logger = Logger.getLogger(PasswordCredentialProvider.class);
    private static final String METER_VALIDATION_OUTCOME_VALID_TAG_VALUE = "valid";
    private static final String METER_VALIDATION_OUTCOME_INVALID_TAG_VALUE = "invalid";
    private static final String METER_VALIDATION_OUTCOME_ERROR_TAG_VALUE = "error";
    protected final KeycloakSession session;
    private final Meter.MeterProvider<Counter> meterProvider;
    private final boolean withAlgorithmInMetric;
    private final boolean metricsEnabled;
    private final boolean withRealmInMetric;
    private final boolean withHashingStrengthInMetric;
    private final boolean withOutcomeInMetric;

    public PasswordCredentialProvider(KeycloakSession session, Meter.MeterProvider<Counter> meterProvider, boolean metricsEnabled, boolean withRealmInMetric, boolean withAlgorithmInMetric, boolean withHashingStrengthInMetric, boolean withOutcomeInMetric) {
        this.session = session;
        this.meterProvider = meterProvider;
        this.metricsEnabled = metricsEnabled;
        this.withRealmInMetric = withRealmInMetric;
        this.withAlgorithmInMetric = withAlgorithmInMetric;
        this.withHashingStrengthInMetric = withHashingStrengthInMetric;
        this.withOutcomeInMetric = withOutcomeInMetric;
    }

    public PasswordCredentialModel getPassword(RealmModel realm, UserModel user) {
        List passwords = user.credentialManager().getStoredCredentialsByTypeStream(this.getType()).collect(Collectors.toList());
        if (passwords.isEmpty()) {
            return null;
        }
        return PasswordCredentialModel.createFromCredentialModel((CredentialModel)((CredentialModel)passwords.get(0)));
    }

    public boolean createCredential(RealmModel realm, UserModel user, String password) {
        PasswordPolicy policy = realm.getPasswordPolicy();
        PolicyError error = ((PasswordPolicyManagerProvider)this.session.getProvider(PasswordPolicyManagerProvider.class)).validate(realm, user, password);
        if (error != null) {
            throw new ModelException(error.getMessage(), error.getParameters());
        }
        PasswordHashProvider hash = this.getHashProvider(policy);
        if (hash == null) {
            return false;
        }
        try {
            PasswordCredentialModel credentialModel = hash.encodedCredential(password, policy.getHashIterations());
            credentialModel.setCreatedDate(Long.valueOf(Time.currentTimeMillis()));
            this.createCredential(realm, user, credentialModel);
        }
        catch (Throwable t) {
            throw new ModelException(t.getMessage(), t);
        }
        return true;
    }

    public CredentialModel createCredential(RealmModel realm, UserModel user, PasswordCredentialModel credentialModel) {
        CredentialModel createdCredential;
        PasswordPolicy policy = realm.getPasswordPolicy();
        int expiredPasswordsPolicyValue = policy.getExpiredPasswords();
        int passwordAgeInDaysPolicy = Math.max(0, policy.getPasswordAgeInDays());
        PasswordCredentialModel oldPassword = this.getPassword(realm, user);
        if (credentialModel.getCreatedDate() == null) {
            credentialModel.setCreatedDate(Long.valueOf(Time.currentTimeMillis()));
        }
        if (oldPassword == null) {
            createdCredential = user.credentialManager().createStoredCredential((CredentialModel)credentialModel);
        } else {
            credentialModel.setId(oldPassword.getId());
            user.credentialManager().updateStoredCredential((CredentialModel)credentialModel);
            createdCredential = credentialModel;
            if (expiredPasswordsPolicyValue > 1 || passwordAgeInDaysPolicy > 0) {
                oldPassword.setId(null);
                oldPassword.setType("password-history");
                oldPassword.setUserLabel(null);
                oldPassword = user.credentialManager().createStoredCredential((CredentialModel)oldPassword);
            }
        }
        int passwordHistoryListMaxSize = Math.max(0, expiredPasswordsPolicyValue - 1);
        long passwordMaxAgeMillis = Time.currentTimeMillis() - Duration.ofDays(passwordAgeInDaysPolicy).toMillis();
        PasswordCredentialModel finalOldPassword = oldPassword;
        user.credentialManager().getStoredCredentialsByTypeStream("password-history").sorted(CredentialModel.comparingByStartDateDesc()).skip(passwordHistoryListMaxSize).filter(arg_0 -> PasswordCredentialProvider.lambda$createCredential$0((CredentialModel)finalOldPassword, arg_0)).filter(credential -> this.passwordAgePredicate((CredentialModel)credential, passwordMaxAgeMillis)).collect(Collectors.toList()).forEach(p -> user.credentialManager().removeStoredCredentialById(p.getId()));
        return createdCredential;
    }

    private boolean passwordAgePredicate(CredentialModel credential, long passwordMaxAgeMillis) {
        long createdDate = credential.getCreatedDate() == null ? Long.MIN_VALUE : credential.getCreatedDate();
        return createdDate < passwordMaxAgeMillis;
    }

    public boolean deleteCredential(RealmModel realm, UserModel user, String credentialId) {
        return user.credentialManager().removeStoredCredentialById(credentialId);
    }

    public PasswordCredentialModel getCredentialFromModel(CredentialModel model) {
        return PasswordCredentialModel.createFromCredentialModel((CredentialModel)model);
    }

    protected PasswordHashProvider getHashProvider(PasswordPolicy policy) {
        if (policy != null && policy.getHashAlgorithm() != null) {
            PasswordHashProvider provider = (PasswordHashProvider)this.session.getProvider(PasswordHashProvider.class, policy.getHashAlgorithm());
            if (provider != null) {
                return provider;
            }
            logger.warnv("Realm PasswordPolicy PasswordHashProvider {0} not found", (Object)policy.getHashAlgorithm());
        }
        return (PasswordHashProvider)this.session.getProvider(PasswordHashProvider.class);
    }

    public boolean supportsCredentialType(String credentialType) {
        return credentialType.equals(this.getType());
    }

    public boolean updateCredential(RealmModel realm, UserModel user, CredentialInput input) {
        return this.createCredential(realm, user, input.getChallengeResponse());
    }

    public void disableCredentialType(RealmModel realm, UserModel user, String credentialType) {
    }

    public Stream<String> getDisableableCredentialTypesStream(RealmModel realm, UserModel user) {
        return Stream.empty();
    }

    public boolean isConfiguredFor(RealmModel realm, UserModel user, String credentialType) {
        return this.getPassword(realm, user) != null;
    }

    public boolean isValid(RealmModel realm, UserModel user, CredentialInput input) {
        if (!(input instanceof UserCredentialModel)) {
            logger.debug((Object)"Expected instance of UserCredentialModel for CredentialInput");
            return false;
        }
        if (input.getChallengeResponse() == null) {
            logger.debugv("Input password was null for user {0} ", (Object)user.getUsername());
            return false;
        }
        PasswordCredentialModel password = this.getPassword(realm, user);
        if (password == null) {
            logger.debugv("No password stored for user {0} ", (Object)user.getUsername());
            return false;
        }
        String algorithm = password.getPasswordCredentialData().getAlgorithm();
        PasswordHashProvider hash = (PasswordHashProvider)this.session.getProvider(PasswordHashProvider.class, algorithm);
        if (hash == null) {
            logger.debugv("PasswordHashProvider {0} not found for user {1} ", (Object)algorithm, (Object)user.getUsername());
            return false;
        }
        try {
            boolean isValid = hash.verify(input.getChallengeResponse(), password);
            if (!isValid) {
                logger.debugv("Failed password validation for user {0} ", (Object)user.getUsername());
                this.publishMetricIfEnabled(realm, algorithm, hash.credentialHashingStrength(password), METER_VALIDATION_OUTCOME_INVALID_TAG_VALUE);
                return false;
            }
            this.rehashPasswordIfRequired(this.session, realm, user, input, password);
        }
        catch (Throwable t) {
            logger.warn((Object)"Error when validating user password", t);
            this.publishMetricIfEnabled(realm, algorithm, hash.credentialHashingStrength(password), METER_VALIDATION_OUTCOME_ERROR_TAG_VALUE);
            return false;
        }
        this.publishMetricIfEnabled(realm, algorithm, hash.credentialHashingStrength(password), METER_VALIDATION_OUTCOME_VALID_TAG_VALUE);
        return true;
    }

    private void publishMetricIfEnabled(RealmModel realm, String algorithm, String hashingStrength, String outcome) {
        if (!this.metricsEnabled) {
            return;
        }
        ArrayList<Tag> tags = new ArrayList<Tag>(5);
        if (this.withAlgorithmInMetric) {
            tags.add(Tag.of((String)"algorithm", (String)this.nullToEmpty(algorithm)));
        }
        if (this.withHashingStrengthInMetric) {
            tags.add(Tag.of((String)"hashing_strength", (String)this.nullToEmpty(hashingStrength)));
        }
        if (this.withRealmInMetric) {
            tags.add(Tag.of((String)"realm", (String)this.nullToEmpty(realm.getName())));
        }
        if (this.withOutcomeInMetric) {
            tags.add(Tag.of((String)"outcome", (String)this.nullToEmpty(outcome)));
        }
        ((Counter)this.meterProvider.withTags(tags)).increment();
    }

    private String nullToEmpty(String value) {
        return value == null ? "" : value;
    }

    private void rehashPasswordIfRequired(final KeycloakSession session, RealmModel realm, final UserModel user, final CredentialInput input, final PasswordCredentialModel password) {
        PasswordPolicy passwordPolicy = realm.getPasswordPolicy();
        PasswordHashProvider provider = passwordPolicy != null && passwordPolicy.getHashAlgorithm() != null ? (PasswordHashProvider)session.getProvider(PasswordHashProvider.class, passwordPolicy.getHashAlgorithm()) : (PasswordHashProvider)session.getProvider(PasswordHashProvider.class);
        if (!provider.policyCheck(passwordPolicy, password)) {
            final int iterations = passwordPolicy != null ? passwordPolicy.getHashIterations() : -1;
            final String hashAlgorithm = passwordPolicy != null ? passwordPolicy.getHashAlgorithm() : null;
            session.getTransactionManager().enlistAfterCompletion((KeycloakTransaction)new AbstractKeycloakTransaction(){

                protected void commitImpl() {
                    try {
                        KeycloakModelUtils.runJobInTransaction((KeycloakSessionFactory)session.getKeycloakSessionFactory(), (KeycloakContext)session.getContext(), s -> PasswordCredentialProvider.refreshPassword(s, hashAlgorithm, iterations, input.getChallengeResponse(), password.getId(), password.getCreatedDate(), password.getUserLabel(), user.getId()));
                    }
                    catch (ModelException e) {
                        logger.info((Object)"Error re-hashing the password in a different transaction", (Throwable)e);
                    }
                }

                protected void rollbackImpl() {
                }
            });
        }
    }

    private static void refreshPassword(KeycloakSession s, String hashAlgorithm, int iterations, String challenge, String passwordId, Long passwordDate, String passwordLabel, String userId) {
        PasswordCredentialModel newPassword = (hashAlgorithm != null ? (PasswordHashProvider)s.getProvider(PasswordHashProvider.class, hashAlgorithm) : (PasswordHashProvider)s.getProvider(PasswordHashProvider.class)).encodedCredential(challenge, iterations);
        newPassword.setId(passwordId);
        newPassword.setCreatedDate(passwordDate);
        newPassword.setUserLabel(passwordLabel);
        UserModel userModel = s.users().getUserById(s.getContext().getRealm(), userId);
        if (userModel != null) {
            userModel.credentialManager().updateStoredCredential((CredentialModel)newPassword);
        }
    }

    public String getType() {
        return "password";
    }

    public CredentialTypeMetadata getCredentialTypeMetadata(CredentialTypeMetadataContext metadataContext) {
        CredentialTypeMetadata.CredentialTypeMetadataBuilder metadataBuilder = CredentialTypeMetadata.builder().type(this.getType()).category(CredentialTypeMetadata.Category.BASIC_AUTHENTICATION).displayName("password-display-name").helpText("password-help-text").iconCssClass("kcAuthenticatorPasswordClass");
        UserModel user = metadataContext.getUser();
        if (user != null && user.credentialManager().isConfiguredFor(this.getType())) {
            metadataBuilder.updateAction(UserModel.RequiredAction.UPDATE_PASSWORD.toString());
        } else {
            metadataBuilder.createAction(UserModel.RequiredAction.UPDATE_PASSWORD.toString());
        }
        return metadataBuilder.removeable(false).build(this.session);
    }

    private static /* synthetic */ boolean lambda$createCredential$0(CredentialModel finalOldPassword, CredentialModel credentialModel1) {
        return !credentialModel1.getId().equals(finalOldPassword.getId());
    }
}

